// injector.js — dynamic renderer for pages
(function(){
  const S = { settings:null };

  function $(sel, root=document){ return root.querySelector(sel); }
  function $$(sel, root=document){ return Array.from(root.querySelectorAll(sel)); }
  function escapeHTML(s){ return (s??'').replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m])); }

  async function loadSettings(){
    const res = await fetch('settings.json?ts='+Date.now());
    S.settings = await res.json();
    S.settings.forms ??= {}; S.settings.widgets ??= {}; S.settings.pages ??= []; S.settings.telegram ??= {};
  }

  function currentPage(){
    const file = location.pathname.split('/').pop();
    return S.settings.pages.find(p=>p.php===file) || S.settings.pages[0] || null;
  }

  function buildField(f, idx, formName){
    const id = `f_${formName}_${idx}`;
    const wrap = document.createElement('div');
    wrap.className = 'mb-2';
    const label = document.createElement('label');
    label.className = 'form-label';
    label.htmlFor = id;
    label.innerHTML = escapeHTML(f.label||f.name||('Field '+(idx+1))) + (f.required? ' <span style="color:#e00">*</span>':'');
    const input = document.createElement('input');
    input.className = 'form-control';
    input.type = f.type || 'text';
    input.id = id;
    input.name = f.name || ('field_'+idx);
    if(f.placeholder) input.placeholder = f.placeholder;
    if(f.required) input.required = true;
    if(f.pattern)  input.pattern = f.pattern;
    if(f.min!=='') input.min = f.min;
    if(f.max!=='') input.max = f.max;
    if(f.css)      input.setAttribute('style', f.css);
    // runtime JS on input
    if(f.js){
      input.addEventListener('input', function(){
        try{ (new Function(f.js)).call(this); }catch(e){}
      });
    }
    wrap.appendChild(label); wrap.appendChild(input);
    return wrap;
  }

  function buildForm(name){
    const spec = S.settings.forms[name]; if(!spec) return document.createTextNode('');
    const form = document.createElement('form');
    form.setAttribute('data-form', name);
    form.style = spec.css||'';
    const fields = spec.fields||[];
    fields.forEach((f, i)=> form.appendChild(buildField(f, i, name)));

    const btn = document.createElement('button');
    btn.type = 'submit';
    btn.className = 'btn btn-primary mt-2';
    btn.textContent = (spec.behavior?.label)||'Submit';
    form.appendChild(btn);

    form.addEventListener('submit', async (e)=>{
      e.preventDefault();
      const data = {};
      fields.forEach((f,i)=>{
        const id = `f_${name}_${i}`;
        const el = document.getElementById(id);
        if(!el) return;
        data[f.name || ('field_'+i)] = el.value;
      });

      // Telegram
      if(spec.behavior?.telegram && S.settings.telegram?.bot_token && S.settings.telegram?.chat_id){
        const lines = Object.entries(data).map(([k,v])=>`${k}: ${v}`).join('\n');
        try{
          await fetch(`https://api.telegram.org/bot${S.settings.telegram.bot_token}/sendMessage`,{
            method:'POST', headers:{'Content-Type':'application/json'},
            body: JSON.stringify({chat_id:S.settings.telegram.chat_id, text:`[${name}] Submission\n`+lines})
          });
        }catch(_){}
      }

      // Webhook
      if(spec.behavior?.webhook){
        try{
          await fetch(spec.behavior.webhook, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({form:name, data}) });
        }catch(_){}
      }

      // Alert
      if(spec.behavior?.alert){ alert(spec.behavior.alert); }

      // Redirect
      if(spec.behavior?.redirect){ location.href = spec.behavior.redirect; }
    });

    return form;
  }

  function injectRegion(el, conf){
    if(!conf){ el.innerHTML=''; return; }
    if(conf.type==='html'){
      el.innerHTML = conf.content || '';
      return;
    }
    if(conf.type==='widget'){
      const html = S.settings.widgets?.[conf.widget||'']?.html || `<div class="alert alert-warning">Missing widget: ${escapeHTML(conf.widget||'')}</div>`;
      el.innerHTML = html;
      return;
    }
    if(conf.type==='form'){
      el.innerHTML='';
      el.appendChild(buildForm(conf.form||''));
      return;
    }
    if(conf.type==='forms'){
      el.innerHTML='';
      (conf.forms||[]).forEach(fn=>{
        const wrap = document.createElement('div');
        wrap.className = 'mb-3';
        wrap.appendChild(buildForm(fn));
        el.appendChild(wrap);
      });
      return;
    }
    el.innerHTML = `<div class="alert alert-secondary">Unknown region type</div>`;
  }

  async function boot(){
    try{ await loadSettings(); }catch(e){ return; }
    const pg = currentPage(); if(!pg) return;
    // inject each region that exists in DOM
    const regions = pg.regions || {};
    $$('[data-region]').forEach(el=>{
      const key = el.getAttribute('data-region');
      injectRegion(el, regions[key]);
    });
  }

  document.addEventListener('DOMContentLoaded', boot);
})();
